-- GrafoConMatrizDeAdyacencia.hs
-- Representación del TAD grafo mediante matriz de adyacencia.
-- José A. Alonso Jiménez <jalonso@us.es>
-- Sevilla, 24 de Octubre de 2010
-- ---------------------------------------------------------------------

module GrafoConMatrizDeAdyacencia 
    (Grafo,
     creaGrafo,  -- (Ix v,Num p) => Bool -> (v,v) -> [(v,v,p)] -> Grafo v p
     adyacentes, -- (Ix v,Num p) => (Grafo v p) -> v -> [v]
     nodos,      -- (Ix v,Num p) => (Grafo v p) -> [v]
     aristasND,  -- (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
     aristasD,   -- (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
     aristaEn,   -- (Ix v,Num p) => (Grafo v p) -> (v,v) -> Bool
     peso        -- (Ix v,Num p) => v -> v -> (Grafo v p) -> p
    ) where

-- ---------------------------------------------------------------------
-- Librerías auxiliares                                               --
-- ---------------------------------------------------------------------

import Data.Array

-- (Grafo v p) es un grafo con vértices de tipo v y pesos de tipo p.
type Grafo v p = Array (v,v) (Maybe p)

-- grafoMA es el grafo
--             12
--        1 -------- 2
--        | \78     /|
--        |  \   32/ |
--        |   \   /  |
--      34|     5    |55
--        |   /   \  |
--        |  /44   \ |
--        | /     93\|
--        3 -------- 4
--             61
-- representado mediante una matriz de adyacencia.
grafoMA = array ((1, 1), (5, 5)) 
                [((1, 1),Nothing), ((1, 2),Just 10), ((1, 3),Just 20), 
                 ((1, 4),Nothing), ((1, 5),Nothing), ((2, 1),Nothing), 
                 ((2, 2),Nothing), ((2, 3),Nothing), ((2, 4),Just 30), 
                 ((2, 5),Nothing), ((3, 1),Nothing), ((3, 2),Nothing), 
                 ((3, 3),Nothing), ((3, 4),Just 40), ((3, 5),Nothing), 
                 ((4, 1),Nothing), ((4, 2),Nothing), ((4, 3),Nothing), 
                 ((4, 4),Nothing), ((4, 5),Just 50), ((5, 1),Nothing), 
                 ((5, 2),Nothing), ((5, 3),Nothing), ((5, 4),Nothing), 
                 ((5, 5),Nothing)]

-- (creaGrafo d cs as) es un grafo (dirigido si de es True y no dirigido
-- en caso contrario), con el par de cotas cs y listas de aristas as
-- (cada arista es un trío formado por los dos vértices y su peso). Ver
-- el ejemplo a continuación.
creaGrafo :: (Ix v, Num p) => Bool -> (v,v) -> [(v,v,p)] -> (Grafo v p)
creaGrafo dir cs@(l,u) as 
    = matrizVacia // 
      ([((x1,x2),Just w) | (x1,x2,w) <- as] ++
       if dir then []
       else [((x2,x1),Just w) | (x1,x2,w) <- as, x1 /= x2])
    where
      matrizVacia = array ((l,l),(u,u)) 
                          [((x1,x2),Nothing) | x1 <- range cs, 
                                               x2 <- range cs]

-- grafoMA' es el mismo grafo que grafoMA pero creado con creaGrafo. Por
-- ejemplo, 
--    ghci> grafoMA'
--    array ((1,1),(5,5)) 
--          [((1,1),Nothing),((1,2),Just 12),((1,3),Just 34),
--           ((1,4),Nothing),((1,5),Just 78),((2,1),Just 12),
--           ((2,2),Nothing),((2,3),Nothing),((2,4),Just 55),
--           ((2,5),Just 32),((3,1),Just 34),((3,2),Nothing),
--           ((3,3),Nothing),((3,4),Just 61),((3,5),Just 44),
--           ((4,1),Nothing),((4,2),Just 55),((4,3),Just 61),
--           ((4,4),Nothing),((4,5),Just 93),((5,1),Just 78),
--           ((5,2),Just 32),((5,3),Just 44),((5,4),Just 93),
--           ((5,5),Nothing)]
grafoMA' = creaGrafo False (1,5) [(1,2,12),(1,3,34),(1,5,78),
                                  (2,4,55),(2,5,32),
                                  (3,4,61),(3,5,44),
                                  (4,5,93)]

-- (adyacentes g v) es la lista de los vértices adyacentes al nodo v en
-- el grafo g. Por ejemplo,
--    adyacentes grafoMA' 4  ==  [2,3,5]
adyacentes :: (Ix v,Num p) => (Grafo v p) -> v -> [v]
adyacentes g v1 = 
    [v2 | v2 <- nodos g, (g!(v1,v2)) /= Nothing]

-- (nodos g) es la lista de todos los nodos del grafo g. Por ejemplo,
--    nodos grafoMA'  ==  [1,2,3,4,5]
nodos :: (Ix v,Num p) => (Grafo v p) -> [v]
nodos g = range (l,u) 
    where ((l,_),(u,_)) = bounds g

-- (aristaEn g a) se verifica si a es una arista del grafo g. Por
-- ejemplo,
--    aristaEn grafoMA' (5,1)  ==  True
--    aristaEn grafoMA' (4,1)  ==  False
aristaEn :: (Ix v,Num p) => (Grafo v p) -> (v,v) -> Bool
aristaEn g (x,y)= (g!(x,y)) /= Nothing

-- (peso v1 v2 g) es el peso de la arista que une los vértices v1 y v2
-- en el grafo g. Por ejemplo,
--    peso 1 5 grafoMA'  ==  78
peso :: (Ix v,Num p) => v -> v -> (Grafo v p) -> p
peso x y g  = w where (Just w) = g!(x,y)

-- (aristasD g) es la lista de las aristas del grafo dirigido g. Por
-- ejemplo, 
--    ghci> aristasD grafoMA'
--    [(1,2,12),(1,3,34),(1,5,78),
--     (2,1,12),(2,4,55),(2,5,32),
--     (3,1,34),(3,4,61),(3,5,44),
--     (4,2,55),(4,3,61),(4,5,93),
--     (5,1,78),(5,2,32),(5,3,44),(5,4,93)]
aristasD :: (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
aristasD g = [(v1,v2,extrae(g!(v1,v2))) 
              | v1 <- nodos g, 
                v2 <- nodos g,
                aristaEn g (v1,v2)]
    where extrae (Just w) = w

-- (aristasND g) es la lista de las aristas del grafo no dirigido g. Por
-- ejemplo, 
--    ghci> aristasND grafoMA'
--    [(1,2,12),(1,3,34),(1,5,78),
--     (2,4,55),(2,5,32),
--     (3,4,61),(3,5,44),
--     (4,5,93)]
aristasND :: (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
aristasND g = [(v1,v2,extrae(g!(v1,v2)))
               | v1 <- nodos g, 
                 v2 <- range (v1,u),
                 aristaEn g (v1,v2)]
    where (_,(u,_)) = bounds g          
          extrae (Just w) = w
