-- GrafoConVectorDeAdyacencia.hs
-- Representación del TAD grafo mediante vectores de adyacencia.
-- José A. Alonso Jiménez <jalonso@us.es>
-- Sevilla, 11 de Octubre de 2010
-- ---------------------------------------------------------------------

module GrafoConVectorDeAdyacencia 
    (Grafo,
     creaGrafo,  -- (Ix v,Num p) => Bool -> (v,v) -> [(v,v,p)] -> Grafo v p
     adyacentes, -- (Ix v,Num p) => (Grafo v p) -> v -> [v]
     nodos,      -- (Ix v,Num p) => (Grafo v p) -> [v]
     aristasND,  -- (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
     aristasD,   -- (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
     aristaEn,   -- (Ix v,Num p) => (Grafo v p) -> (v,v) -> Bool
     peso        -- (Ix v,Num p) => v -> v -> (Grafo v p) -> p
    ) where

-- ---------------------------------------------------------------------
-- Librerías auxiliares                                               --
-- ---------------------------------------------------------------------

import Data.Array

-- (Grafo v p) es un grafo con vértices de tipo v y pesos de tipo p.
type Grafo v p = Array v [(v,p)]

-- grafoVA es el grafo
--             12
--        1 -------- 2
--        | \78     /|
--        |  \   32/ |
--        |   \   /  |
--      34|     5    |55
--        |   /   \  |
--        |  /44   \ |
--        | /     93\|
--        3 -------- 4
--             61
-- representado mediante un vector de adyacencia.
grafoVA = array (1,5) [(1,[(2,12),(3,34),(5,78)]),
                       (2,[(1,12),(4,55),(5,32)]),
                       (3,[(1,34),(4,61),(5,44)]),
                       (4,[(2,55),(3,61),(5,93)]),
                       (5,[(1,78),(2,32),(3,44),(4,93)])]

-- (creaGrafo d cs as) es un grafo (dirigido si d es True y no dirigido
-- en caso contrario), con el par de cotas cs y listas de aristas as
-- (cada arista es un trío formado por los dos vértices y su peso). Ver
-- el ejemplo a continuación.
creaGrafo :: (Ix v, Num p) => Bool -> (v,v) -> [(v,v,p)] -> Grafo v p
creaGrafo d cs vs =
    accumArray 
     (\xs x -> xs++[x]) [] cs 
     ((if d then []
       else [(x2,(x1,p))|(x1,x2,p) <- vs, x1 /= x2]) ++
      [(x1,(x2,p)) | (x1,x2,p) <- vs])

-- grafoVA' es el mismo grafo que grafoVA pero creado con creaGrafo. Por
-- ejemplo, 
--    ghci> grafoVA'
--    array (1,5) [(1,[(2,12),(3,34),(5,78)]),
--                 (2,[(1,12),(4,55),(5,32)]),
--                 (3,[(1,34),(4,61),(5,44)]),
--                 (4,[(2,55),(3,61),(5,93)]),
--                 (5,[(1,78),(2,32),(3,44),(4,93)])]
grafoVA' = creaGrafo False (1,5) [(1,2,12),(1,3,34),(1,5,78),
                                  (2,4,55),(2,5,32),
                                  (3,4,61),(3,5,44),
                                  (4,5,93)]

-- (adyacentes g v) es la lista de los vértices adyacentes al nodo v en
-- el grafo g. Por ejemplo,
--    adyacentes grafoVA' 4  ==  [2,3,5]
adyacentes :: (Ix v,Num p) => (Grafo v p) -> v -> [v]
adyacentes g v = map fst (g!v)

-- (nodos g) es la lista de todos los nodos del grafo g. Por ejemplo,
--    nodos grafoVA'  ==  [1,2,3,4,5]
nodos :: (Ix v,Num p) => (Grafo v p) -> [v]
nodos g = indices g

-- (aristaEn g a) se verifica si a es una arista del grafo g. Por
-- ejemplo,
--    aristaEn grafoVA' (5,1)  ==  True
--    aristaEn grafoVA' (4,1)  ==  False
aristaEn :: (Ix v,Num p) => (Grafo v p) -> (v,v) -> Bool
aristaEn g (x,y) = elem y (adyacentes g x)

-- (peso v1 v2 g) es el peso de la arista que une los vértices v1 y v2
-- en el grafo g. Por ejemplo,
--    peso 1 5 grafoVA'  ==  78
peso :: (Ix v,Num p) => v -> v -> (Grafo v p) -> p
peso x y g = head [c | (a,c) <- g!x , a == y]

-- (aristasD g) es la lista de las aristas del grafo dirigido g. Por
-- ejemplo, 
--    ghci> aristasD grafoVA'
--    [(1,2,12),(1,3,34),(1,5,78),
--     (2,1,12),(2,4,55),(2,5,32),
--     (3,1,34),(3,4,61),(3,5,44),
--     (4,2,55),(4,3,61),(4,5,93),
--     (5,1,78),(5,2,32),(5,3,44),(5,4,93)]
aristasD :: (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
aristasD g = [(v1,v2,w) | v1 <- nodos g , (v2,w) <- g!v1] 

-- (aristasND g) es la lista de las aristas del grafo no dirigido g. Por
-- ejemplo, 
--    ghci> aristasND grafoVA'
--    [(1,2,12),(1,3,34),(1,5,78),
--     (2,4,55),(2,5,32),
--     (3,4,61),(3,5,44),
--     (4,5,93)]
aristasND :: (Ix v,Num p) => (Grafo v p) -> [(v,v,p)]
aristasND g = 
    [(v1,v2,w) | v1 <- nodos g , (v2,w) <- g!v1 , v1 < v2] 

-- La complejidad de las funciones adyacente y peso es de O(|V|).

